// Copyright 2019-2020 CERN and copyright holders of ALICE O2.
// See https://alice-o2.web.cern.ch/copyright for details of the copyright holders.
// All rights not expressly granted are reserved.
//
// This software is distributed under the terms of the GNU General Public
// License v3 (GPL Version 3), copied verbatim in the file "COPYING".
//
// In applying this license CERN does not waive the privileges and immunities
// granted to it by virtue of its status as an Intergovernmental Organization
// or submit itself to any jurisdiction.

// C++/ROOT includes.
#include <chrono>
#include <string>
#include <vector>
#include <TComplex.h>
#include <TH1F.h>
#include <TH2D.h>
#include <TMath.h>
#include <TVector2.h>

// o2Physics includes.
#include "Framework/AnalysisDataModel.h"
#include "Framework/AnalysisTask.h"
#include "Framework/ASoAHelpers.h"
#include "Framework/HistogramRegistry.h"
#include "Framework/runDataProcessing.h"
#include "Framework/RunningWorkflowInfo.h"
#include "Framework/StaticFor.h"

#include "Common/DataModel/Qvectors.h"
#include "Common/DataModel/EventSelection.h"
#include "Common/DataModel/TrackSelectionTables.h"
#include "Common/DataModel/Centrality.h"
#include "Common/Core/EventPlaneHelper.h"
#include "Common/Core/TrackSelection.h"

#include "CommonConstants/PhysicsConstants.h"

#include "CCDB/BasicCCDBManager.h"
#include "CCDB/CcdbApi.h"

// o2 includes.

using namespace o2;
using namespace o2::framework;

using MyCollisions = soa::Join<aod::Collisions, aod::EvSels, aod::CentFT0Cs, aod::QvectorFT0CVecs, aod::QvectorTPCposVecs, aod::QvectorTPCnegVecs>;
using MyTracks = soa::Join<aod::Tracks, aod::TracksExtra, aod::TrackSelection, aod::TrackSelectionExtension>;

struct qVectorstutorial {
  HistogramRegistry histosQA{"histosQA", {}, OutputObjHandlingPolicy::AnalysisObject, false, false};

  Configurable<std::vector<int>> cfgnMods{"cfgnMods", {2}, "Modulation of interest"};
  Configurable<std::string> cfgDetName{"cfgDetName", "FT0C", "The name of detector to be analyzed"};
  Configurable<std::string> cfgRefAName{"cfgRefAName", "TPCpos", "The name of detector for reference A"};
  Configurable<std::string> cfgRefBName{"cfgRefBName", "TPCneg", "The name of detector for reference B"};

  Configurable<bool> cfgShiftCorr{"cfgShiftCorr", false, "additional shift correction"};
  Configurable<std::string> cfgShiftPath{"cfgShiftPath", "Users/j/junlee/lambdapol_shiftcor/Pass2", "Path for Shift"};

  Configurable<float> cfgMinPt{"cfgMinPt", 0.15, "Minimum transverse momentum for charged track"};
  Configurable<float> cfgMaxEta{"cfgMaxEta", 0.8, "Maximum pseudorapidiy for charged track"};
  Configurable<float> cfgMaxDCArToPVcut{"cfgMaxDCArToPVcut", 0.1, "Maximum transverse DCA"};
  Configurable<float> cfgMaxDCAzToPVcut{"cfgMaxDCAzToPVcut", 1.0, "Maximum longitudinal DCA"};

  ConfigurableAxis cfgaxisQvecF{"cfgaxisQvecF", {300, -1, 1}, ""};
  ConfigurableAxis cfgaxisQvec{"cfgaxisQvec", {100, -3, 3}, ""};
  ConfigurableAxis cfgaxisCent{"cfgaxisCent", {100, 0, 100}, ""};

  ConfigurableAxis cfgaxiscos{"cfgaxiscos", {102, -1.02, 1.02}, ""};
  ConfigurableAxis cfgaxispt{"cfgaxispt", {100, 0, 10}, ""};
  ConfigurableAxis cfgaxisCentMerged{"cfgaxisCentMerged", {20, 0, 100}, ""};

  EventPlaneHelper helperEP;

  Service<o2::ccdb::BasicCCDBManager> ccdb;
  o2::ccdb::CcdbApi ccdbApi;

  int currentRunNumber = -999;
  int lastRunNumber = -999;

  std::vector<TProfile3D*> shiftprofile{};
  std::string fullCCDBShiftCorrPath;

  void init(InitContext const&)
  {
    AxisSpec axisCent{cfgaxisCent, "centrality"};
    AxisSpec axisQvec{cfgaxisQvec, "Q"};
    AxisSpec axisQvecF{cfgaxisQvecF, "Q"};
    AxisSpec axisEvtPl = {100, -1.0 * constants::math::PI, constants::math::PI};

    AxisSpec axisCos{cfgaxiscos, "angle function"};
    AxisSpec axisPt{cfgaxispt, "trasverse momentum"};
    AxisSpec axisCentMerged{cfgaxisCentMerged, "merged centrality"};

    histosQA.add(Form("histQvecV2"), "", {HistType::kTH3F, {axisQvecF, axisQvecF, axisCent}});
    histosQA.add(Form("histEvtPlV2"), "", {HistType::kTH2F, {axisEvtPl, axisCent}});
    histosQA.add(Form("histQvecRes_SigRefAV2"), "", {HistType::kTH2F, {axisQvecF, axisCent}});
    histosQA.add(Form("histCosDetV2"), "", {HistType::kTH3F, {axisCentMerged, axisPt, axisCos}});

    histosQA.add(Form("histCorEvtPlV2"), "", {HistType::kTH2F, {axisEvtPl, axisCent}});
    histosQA.add(Form("histCorCosDetV2"), "", {HistType::kTH3F, {axisCentMerged, axisPt, axisCos}});

    histosQA.add("EpResQvecDetRefAxx", "", {HistType::kTH2F, {axisCentMerged, axisQvec}});
    histosQA.add("EpResQvecDetRefBxx", "", {HistType::kTH2F, {axisCentMerged, axisQvec}});
    histosQA.add("EpResQvecRefARefBxx", "", {HistType::kTH2F, {axisCentMerged, axisQvec}});

    histosQA.add("SPvnxx", "", {HistType::kTH3F, {axisCentMerged, axisPt, axisQvec}});
  }

  template <typename CollType>
  bool SelEvent(const CollType& collision)
  {
    if (!collision.sel8()) {
      return 0;
    }
    if (!collision.selection_bit(aod::evsel::kIsGoodZvtxFT0vsPV)) {
      return 0;
    }
    if (!collision.selection_bit(aod::evsel::kNoSameBunchPileup)) {
      return 0;
    }
    if (!collision.selection_bit(o2::aod::evsel::kNoCollInTimeRangeStandard)) {
      return 0;
    }

    return 1;
  }

  template <typename TrackType>
  bool SelTrack(const TrackType track)
  {
    if (track.pt() < cfgMinPt)
      return false;
    if (std::abs(track.eta()) > cfgMaxEta)
      return false;
    if (!track.passedITSNCls())
      return false;
    if (!track.passedITSChi2NDF())
      return false;
    if (!track.passedITSHits())
      return false;
    if (!track.passedTPCCrossedRowsOverNCls())
      return false;
    if (!track.passedTPCChi2NDF())
      return false;
    if (!track.passedDCAxy())
      return false;
    if (!track.passedDCAz())
      return false;

    return true;
  }

  template <typename CollType>
  void ShiftCorrection(const CollType& collision, std::vector<float>& QvecReCorr, std::vector<float>& QvecImCorr, int nmode)
  {
    if (nmode == 2) {
      auto psidefFT0C = helperEP.GetEventPlane(collision.qvecFT0CReVec()[0], collision.qvecFT0CImVec()[0], nmode);
      auto psidefTPCf = helperEP.GetEventPlane(collision.qvecTPCposReVec()[0], collision.qvecTPCposImVec()[0], nmode);
      auto psidefTPCb = helperEP.GetEventPlane(collision.qvecTPCnegReVec()[0], collision.qvecTPCnegImVec()[0], nmode);

      auto deltapsiFT0C = 0.0;
      auto deltapsiTPCf = 0.0;
      auto deltapsiTPCb = 0.0;

      for (int ishift = 1; ishift <= 10; ishift++) {
        auto coeffshiftxFT0C = shiftprofile.at(nmode - 2)->GetBinContent(shiftprofile.at(nmode - 2)->FindBin(collision.centFT0C(), 0.5, ishift - 0.5));
        auto coeffshiftyFT0C = shiftprofile.at(nmode - 2)->GetBinContent(shiftprofile.at(nmode - 2)->FindBin(collision.centFT0C(), 1.5, ishift - 0.5));
        auto coeffshiftxTPCf = shiftprofile.at(nmode - 2)->GetBinContent(shiftprofile.at(nmode - 2)->FindBin(collision.centFT0C(), 2.5, ishift - 0.5));
        auto coeffshiftyTPCf = shiftprofile.at(nmode - 2)->GetBinContent(shiftprofile.at(nmode - 2)->FindBin(collision.centFT0C(), 3.5, ishift - 0.5));
        auto coeffshiftxTPCb = shiftprofile.at(nmode - 2)->GetBinContent(shiftprofile.at(nmode - 2)->FindBin(collision.centFT0C(), 4.5, ishift - 0.5));
        auto coeffshiftyTPCb = shiftprofile.at(nmode - 2)->GetBinContent(shiftprofile.at(nmode - 2)->FindBin(collision.centFT0C(), 5.5, ishift - 0.5));

        deltapsiFT0C += ((2 / (1.0 * ishift)) * (-coeffshiftxFT0C * TMath::Cos(ishift * static_cast<float>(nmode) * psidefFT0C) + coeffshiftyFT0C * TMath::Sin(ishift * static_cast<float>(nmode) * psidefFT0C)));
        deltapsiTPCf += ((2 / (1.0 * ishift)) * (-coeffshiftxTPCf * TMath::Cos(ishift * static_cast<float>(nmode) * psidefTPCf) + coeffshiftyTPCf * TMath::Sin(ishift * static_cast<float>(nmode) * psidefTPCf)));
        deltapsiTPCb += ((2 / (1.0 * ishift)) * (-coeffshiftxTPCb * TMath::Cos(ishift * static_cast<float>(nmode) * psidefTPCb) + coeffshiftyTPCb * TMath::Sin(ishift * static_cast<float>(nmode) * psidefTPCb)));
      }
      QvecReCorr.push_back(collision.qvecFT0CReVec()[0] * std::cos(deltapsiFT0C) - collision.qvecFT0CImVec()[0] * std::sin(deltapsiFT0C));
      QvecReCorr.push_back(collision.qvecTPCposReVec()[0] * std::cos(deltapsiTPCf) - collision.qvecTPCposImVec()[0] * std::sin(deltapsiTPCf));
      QvecReCorr.push_back(collision.qvecTPCnegReVec()[0] * std::cos(deltapsiTPCb) - collision.qvecTPCnegImVec()[0] * std::sin(deltapsiTPCb));

      QvecImCorr.push_back(collision.qvecFT0CReVec()[0] * std::sin(deltapsiFT0C) + collision.qvecFT0CImVec()[0] * std::cos(deltapsiFT0C));
      QvecImCorr.push_back(collision.qvecTPCposReVec()[0] * std::sin(deltapsiTPCf) + collision.qvecTPCposImVec()[0] * std::cos(deltapsiTPCf));
      QvecImCorr.push_back(collision.qvecTPCnegReVec()[0] * std::sin(deltapsiTPCb) + collision.qvecTPCnegImVec()[0] * std::cos(deltapsiTPCb));
    }
  }

  template <typename CollType>
  void fillHistosQvec(const CollType& collision, int nmode)
  {
    if (nmode == 2) {
      histosQA.fill(HIST("histQvecV2"), collision.qvecFT0CReVec()[0], collision.qvecFT0CImVec()[0], collision.centFT0C());
      histosQA.fill(HIST("histEvtPlV2"), helperEP.GetEventPlane(collision.qvecFT0CReVec()[0], collision.qvecFT0CImVec()[0], nmode), collision.centFT0C());
      histosQA.fill(HIST("histQvecRes_SigRefAV2"), helperEP.GetResolution(helperEP.GetEventPlane(collision.qvecFT0CReVec()[0], collision.qvecFT0CImVec()[0], nmode), helperEP.GetEventPlane(collision.qvecTPCposReVec()[0], collision.qvecTPCposImVec()[0], nmode), nmode), collision.centFT0C());
      if (cfgShiftCorr) {
        std::vector<float> qvecRe{};
        std::vector<float> qvecIm{};
        ShiftCorrection(collision, qvecRe, qvecIm, nmode);

        histosQA.fill(HIST("histCorEvtPlV2"), helperEP.GetEventPlane(qvecRe[0], qvecIm[0], nmode), collision.centFT0C());

        histosQA.fill(HIST("EpResQvecDetRefAxx"), collision.centFT0C(), qvecRe[0] * qvecRe[1] + qvecIm[0] * qvecIm[1]);
        histosQA.fill(HIST("EpResQvecDetRefBxx"), collision.centFT0C(), qvecRe[0] * qvecRe[2] + qvecIm[0] * qvecIm[2]);
        histosQA.fill(HIST("EpResQvecRefARefBxx"), collision.centFT0C(), qvecRe[1] * qvecRe[2] + qvecIm[1] * qvecIm[2]);
      }
    }
  }

  template <typename CollType, typename TrackType>
  void fillHistosFlow(const CollType& collision, const TrackType& track, int nmode)
  {
    if (collision.sumAmplFT0C() < 1e-4) {
      return;
    }
    for (auto& trk : track) {
      if (!SelTrack(trk)) {
        continue;
      }
      if (nmode == 2) {
        histosQA.fill(HIST("histCosDetV2"), collision.centFT0C(), trk.pt(),
                      std::cos(static_cast<float>(nmode) * (trk.phi() - helperEP.GetEventPlane(collision.qvecFT0CReVec()[0], collision.qvecFT0CImVec()[0], nmode))));
        if (cfgShiftCorr) {
          std::vector<float> qvecRe{};
          std::vector<float> qvecIm{};
          ShiftCorrection(collision, qvecRe, qvecIm, nmode);

          histosQA.fill(HIST("histCorCosDetV2"), collision.centFT0C(), trk.pt(),
                       std::cos(static_cast<float>(nmode) * (trk.phi() - helperEP.GetEventPlane(qvecRe[0], qvecIm[0], nmode))));

          histosQA.fill(HIST("SPvnxx"), collision.centFT0C(), trk.pt(), std::cos(static_cast<float>(nmode) * trk.phi()) * qvecRe[0] + std::sin(static_cast<float>(nmode) * trk.phi()) * qvecIm[0], 2);
        }
      }
    }
  }

  void process(MyCollisions::iterator const& collision, MyTracks const& tracks, aod::BCsWithTimestamps const&)
  {
    if (!SelEvent(collision)) {
      return;
    }

    if (cfgShiftCorr) {
      auto bc = collision.bc_as<aod::BCsWithTimestamps>();
      currentRunNumber = bc.runNumber();
      if (currentRunNumber != lastRunNumber) {
        shiftprofile.clear();
        for (std::size_t i = 0; i < cfgnMods->size(); i++) {
          fullCCDBShiftCorrPath = cfgShiftPath;
          fullCCDBShiftCorrPath += "/v";
          fullCCDBShiftCorrPath += std::to_string(cfgnMods->at(i));
          auto objshift = ccdb->getForTimeStamp<TProfile3D>(fullCCDBShiftCorrPath, bc.timestamp());
          shiftprofile.push_back(objshift);
        }
        lastRunNumber = currentRunNumber;
      }
    }

    for (std::size_t i = 0; i < cfgnMods->size(); i++) {
      fillHistosQvec(collision, cfgnMods->at(i));
      fillHistosFlow(collision, tracks, cfgnMods->at(i));
    }
  }
};

WorkflowSpec defineDataProcessing(ConfigContext const& cfgc)
{
  return WorkflowSpec{
    adaptAnalysisTask<qVectorstutorial>(cfgc)};
}
